/**
* \file: msd.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <stdio.h>
#include <libudev.h>
#include <sys/epoll.h>
#include <sys/mount.h>
#include <sys/stat.h>
#include <errno.h>
#include <sys/unistd.h>
#include <stdbool.h>
#include <stdlib.h>
#include <string.h>
#include <linux/limits.h>
#include <alloca.h>

#include "utils/logger.h"
#include "backends/msd_events.h"
#include "backends/msd.h"
#include "backends/msd-metadata.h"
#include "model/device_list.h"
#include "model/partition_list.h"
#include "control/device_fsm.h"
#include "control/partition_fsm.h"
#include "control/mount.h"
#include "control/blacklist.h"
#include "control/configuration.h"
#include "control/automounter.h"
#include "app_iface/app_iface_intern.h"

//vtable members
static error_code_t msd_init(void);
static error_code_t msd_blacklist_mounted_device(const char *mount_src);
static void msd_deinit(void);

//signals
static void msd_on_watch_event(watch_t *ev_watch, uint32_t events);
static void msd_signal_device_added(msd_uevent_info_cache_t *uevent_info_cache);
static void msd_signal_device_removed(msd_uevent_info_cache_t *uevent_info_cache);
static void msd_signal_device_media_inserted(msd_uevent_info_cache_t *uevent_info_cache);
static void msd_signal_device_media_removed(msd_uevent_info_cache_t *uevent_info_cache);

static void msd_signal_partition_added(msd_uevent_info_cache_t *uevent_info_cache,
		device_t *device);
static void msd_signal_partition_removed(msd_uevent_info_cache_t *uevent_info_cache);

//state members
static void msd_enter_device_detected(msd_uevent_info_cache_t *uevent_info_cache);
static void msd_enter_read_out_device_info(msd_uevent_info_cache_t *uevent_info_cache,
		device_t *device);
static void msd_enter_partition_detected(device_t *device,
		msd_uevent_info_cache_t *uevent_info_cache);
static void msd_enter_partition_signal_dev_added(msd_uevent_info_cache_t *dev_uevent_info_cache,
		msd_uevent_info_cache_t *part_uevent_info_cache);
static void msd_enter_partition_signal_media_inserted(msd_uevent_info_cache_t *dev_uevent_info_cache,
		msd_uevent_info_cache_t *part_uevent_info_cache);

//others
static void msd_prepare_device_changed_events(msd_uevent_info_cache_t *uevent_info_cache);
static bool msd_is_media_change_event(msd_uevent_info_cache_t *uevent_info_cache);
static bool msd_is_media_eject_complete_event(msd_uevent_info_cache_t *uevent_info_cache);
static bool msd_device_contains_media(msd_uevent_info_cache_t *uevent_info_cache);
static device_t *msd_find_device_in_model(msd_uevent_info_cache_t *uevent_info_cache);
static partition_t *msd_find_partition_in_model(msd_uevent_info_cache_t *uevent_info_cache);
static partition_unsupported_reason_t msd_prepare_partition(device_t *device, partition_t **partition_ptr,
		msd_uevent_info_cache_t *uevent_info_cache);
static  error_code_t msd_create_mount_point(char *mount_point, size_t mount_point_buffer_size,
		msd_uevent_info_cache_t *uevent_info_cache);

//singleton attributes of msd device handler
static struct udev *udev=NULL;
static struct udev_monitor *monitor=NULL;
static watch_t msd_watch={-1,0,msd_on_watch_event,NULL};

//-------------------------------------------- vtable definition -------------------------------------------
devicehandler_vtable_t msd_handler_vtable =
{
		.init=msd_init,
		.blacklist_mounted_device=msd_blacklist_mounted_device,
		.deinit=msd_deinit,
		.device_handler_id=GENERAL_MSD_DEVICE
};
//----------------------------------------------------------------------------------------------------------


//-------------------------------- MSD Handler vtable members ----------------------------------------------------
static  error_code_t msd_init(void)
{    
	error_code_t result=RESULT_OK;
	logger_log_debug("MSD_Handler - Initializing the mass storage device handler.");

	monitor=NULL;

	udev = udev_new();
	if (udev==NULL)
		result=RESULT_NORESOURCE;
	else
		monitor = udev_monitor_new_from_netlink(udev, "udev");

	if (!monitor)
		result=RESULT_NORESOURCE;

	if (result==RESULT_OK)
	{
		if (udev_monitor_enable_receiving(monitor))
			result=RESULT_NORESOURCE;
	}

	if (result==RESULT_OK)
	{
		if (udev_monitor_filter_add_match_subsystem_devtype(monitor, "block", NULL))
			result=RESULT_NORESOURCE;
	}

	if (result==RESULT_OK)
	{
		msd_watch.pollfd=udev_monitor_get_fd(monitor);
		result=watch_add_event_source(&msd_watch,EPOLLIN);
	}

	if (result!=RESULT_OK)
		msd_deinit();

	return result;
}

static void msd_deinit(void)
{  
	if (msd_watch.pollfd!=-1)
	{
		(void)watch_remove_event_source(&msd_watch);
		msd_watch.pollfd=-1;
	}

	if (monitor!=NULL)
	{
		udev_monitor_unref(monitor);
		monitor=NULL;
	}

	if (udev!=NULL)
	{
		udev_unref(udev);
		udev=NULL;
	}
	logger_log_debug("MSD_Handler - Deinitializing the mass storage device handler.");
}

static error_code_t msd_blacklist_mounted_device(const char *mount_src)
{
	struct udev_device *upartition;
	struct udev_device *udevice;

	upartition=msd_metadata_find_udevice_by_devnode(udev,mount_src);
	if (upartition!=NULL)
	{
		if (strcmp(udev_device_get_subsystem(upartition),"block")==0)
			blacklist_add_blacklisted_partition(mount_src);

		udevice=msd_metadata_determine_parent_device(upartition);
		if (udevice!=NULL)
			//we have a partition mounted, add the underlying device to the blacklist
			blacklist_add_blacklisted_device(udev_device_get_devnode(udevice));
		else
			//we mounted a device without partitions (e.g. CDROM). Add it to the device blacklist as well
			blacklist_add_blacklisted_device(mount_src);
	}

	return RESULT_OK;
}
//--------------------------------------------------------------------------------------------------------

//--------------------------------------signals ----------------------------------------------------------
//event dispatcher
static void msd_on_watch_event(watch_t *ev_watch, uint32_t events)
{
	struct udev_device  *dev_of_event;
	const char *action, *dev_type;
	const char *devnode;
	msd_uevent_info_cache_t *uevent_info_cache;

	//we know our watch
	(void)ev_watch;
	//we are registered only for POLLIN
	(void)events;

	dev_of_event = udev_monitor_receive_device(monitor);

	if (!dev_of_event)
		return;

	action = udev_device_get_action(dev_of_event);

	uevent_info_cache=(msd_uevent_info_cache_t*)alloca(msd_uevent_get_size_cache());
	if(uevent_info_cache==NULL)
		automounter_die_on_resource_issues();
	msd_uevent_init(dev_of_event, uevent_info_cache);

	dev_type = msd_uevent_get_dev_type(uevent_info_cache);

	if (!strcmp(dev_type,"disk"))
	{
		//we are a device
		devnode=msd_uevent_get_dev_node(uevent_info_cache);
		if (blacklist_is_device_blacklisted(devnode))
			logger_log_info("MSD_Handler - Received event from blacklisted device: %s. Ignoring it.",devnode);
		else if (!strcmp(action, "add"))
			msd_signal_device_added(uevent_info_cache);
		else if (!strcmp(action, "remove"))
			msd_signal_device_removed(uevent_info_cache);
		else if (!strcmp(action, "change"))
			msd_prepare_device_changed_events(uevent_info_cache);
		else
			logger_log_error("MSD_Handler - Uevent with unknown action received: %s. Ignoring it.",action);
	}
	else if (!strcmp(dev_type,"partition"))
	{
		//we are a partition
		devnode=msd_uevent_get_dev_node(uevent_info_cache);
		if (blacklist_is_partition_blacklisted(devnode))
			logger_log_info("MSD_Handler - Received event from blacklisted device: %s. Ignoring it.",devnode);
		if (!strcmp(action, "add"))
			msd_signal_partition_added(uevent_info_cache,NULL);
		else if (!strcmp(action, "remove"))
			msd_signal_partition_removed(uevent_info_cache);
		else if (!strcmp(action, "change"))
			logger_log_debug("MSD_Handler - Change uevent received for partition: %s. Ignoring it.",devnode);
		else
			logger_log_error("MSD_Handler - Uevent with unknown action received: %s. Ignoring it.",action);
	}
	else
	{
		logger_log_error("MSD_Handler - Received uevent with unexpected type.");
	}

	msd_uevent_deinit(uevent_info_cache);
	udev_device_unref(dev_of_event);
}

static void msd_signal_device_added(msd_uevent_info_cache_t *uevent_info_cache)
{
	const char* dev_dev_node;
	dev_dev_node=msd_uevent_get_dev_node(uevent_info_cache);
	logger_log_debug("MSD_Handler - Event received: added device %s.", dev_dev_node);

	if (device_list_find_device(dev_dev_node)==NULL)
	{
		//signal: dev_added
		if (!blacklist_is_device_blacklisted(dev_dev_node))
			//guard: not blacklisted
			msd_enter_device_detected(uevent_info_cache);
		else
			logger_log_info("MSD_Handler - Device %s blacklisted. Ignoring the event.",dev_dev_node);
	}
	else
		//Add device event received twice. Ignoring the event
		logger_log_info("MSD_Handler - Device %s already added before. Ignoring the event.",dev_dev_node);
}

static void msd_signal_device_removed(msd_uevent_info_cache_t *uevent_info_cache)
{
	device_t *device;
	device=msd_find_device_in_model(uevent_info_cache);
	if (device!=NULL)
		device_fsm_signal_device_removed(device);
	else
		logger_log_info("MSD_Handler - Got a remove event for a device that is not handled by us. Ignoring it.");
}

static void msd_signal_device_media_inserted(msd_uevent_info_cache_t *uevent_info_cache)
{
	device_state_t coming_from;
	device_t *device;
	device=msd_find_device_in_model(uevent_info_cache);
	if (device!=NULL)
	{
		coming_from=device_get_state(device);
		if (coming_from==DEVICE_NOMEDIA)
			msd_enter_read_out_device_info(uevent_info_cache,device);
	}
	else
		logger_log_info("MSD_Handler - Got a media insert event for a device that is not handled by us. Ignoring it.");
}

static void msd_signal_device_media_removed(msd_uevent_info_cache_t *uevent_info_cache)
{  
	device_t *device;
	device=msd_find_device_in_model(uevent_info_cache);
	if (device!=NULL)
	{
		if (device_get_state(device)!=DEVICE_NOMEDIA)
			device_fsm_signal_media_removed(device);
	}
	else
		logger_log_info("MSD_Handler - Got a media remove event for a device that is not handled by us. Ignoring it.");
}

static void msd_signal_partition_added(msd_uevent_info_cache_t *part_uevent_info_cache, device_t *device)
{
	msd_uevent_info_cache_t *dev_uevent_info_cache=NULL;
	struct udev_device *udevice=NULL;
	const char* part_dev_node;

	part_dev_node=msd_uevent_get_dev_node(part_uevent_info_cache);
	logger_log_debug("MSD_Handler - Event received: added partition %s.", part_dev_node);

	if (device_list_find_partition_by_id(part_dev_node)==NULL)
	{
		if (device==NULL)
		{
			udevice=msd_uevent_determine_parent_device(part_uevent_info_cache);
		}

		if (udevice!=NULL || device!=NULL)
		{
			if (device==NULL)
			{
				dev_uevent_info_cache=(msd_uevent_info_cache_t*)alloca(msd_uevent_get_size_cache());
				if(dev_uevent_info_cache==NULL)
					automounter_die_on_resource_issues();
				msd_uevent_init(udevice,dev_uevent_info_cache);
				device=msd_find_device_in_model(dev_uevent_info_cache);
			}

			if (device!=NULL)
			{
				device_state_t dev_state=device_get_state(device);
				if (dev_state==DEVICE_AUTOMOUNTING || dev_state==DEVICE_AUTOMOUNTED)
					msd_enter_partition_detected(device,part_uevent_info_cache);
				else if (dev_state==DEVICE_NOMEDIA)
					msd_enter_partition_signal_media_inserted(dev_uevent_info_cache,part_uevent_info_cache);
				else
					logger_log_error("MSD_Handler - Received add event of partition %s. This partition belongs to a device"
							" that has been unmounted before or is currently being unmounted. Ignoring the event.",part_dev_node);
			}
			else
			{
				if (!blacklist_is_device_blacklisted(msd_uevent_get_dev_node(dev_uevent_info_cache)))
					msd_enter_partition_signal_dev_added(dev_uevent_info_cache,part_uevent_info_cache);
				else
					logger_log_info("MSD_Handler - The device that belongs to partition %s is blacklisted."
							" Ignoring the partition.",part_dev_node);
			}

			//it might be the case that noone has initialized the cache before because it wasn't used. So just check it
			//before we are deinitializing it
			if (dev_uevent_info_cache!=NULL)
				msd_uevent_deinit(dev_uevent_info_cache);
		}
		else
			logger_log_error("MSD_Handler - Unable to determine the parent device of partition %s."
					"Maybe the device is already gone. Ignoring the event.",part_dev_node);
	}
	else
		//Add partition event received twice. Ignoring the event
		logger_log_info("MSD_Handler - Partition %s already added before. Ignoring the event.",part_dev_node);
}

static void msd_signal_partition_removed(msd_uevent_info_cache_t *uevent_info_cache)
{
	partition_t *partition;
	partition=msd_find_partition_in_model(uevent_info_cache);
	if (partition!=NULL)
		partition_fsm_signal_part_removed(partition);
	else
		logger_log_info("MSD_Handler - Got a remove event for partition that is not handled by us. Ignoring it.");
}
//--------------------------------------------------------------------------------------------------------

//-------------------------------- state members ---------------------------------------------------------
static void msd_enter_device_detected(msd_uevent_info_cache_t *uevent_info_cache)
{
	error_code_t result;
	const char* dev_dev_node;
	device_t *device;
	device_metadata_t *metadata;

	result=msd_metadata_create_for_device(&metadata,uevent_info_cache);

	if (result==RESULT_OK)
	{
		dev_dev_node=msd_uevent_get_dev_node(uevent_info_cache);
		device=device_list_add_device(dev_dev_node, metadata,&msd_handler_vtable);
		if (device==NULL)
			automounter_die_on_resource_issues();

		app_iface_signal_device_detected(device);
		logger_log_debug("MSD_Handler - Device %s: in state detected",device_get_id(device));
		if (msd_device_contains_media(uevent_info_cache))
			msd_enter_read_out_device_info(uevent_info_cache,device);
		else
			device_fsm_enter_device_nomedia(device);
	}
	else if (result==RESULT_NORESOURCE)
		automounter_die_on_resource_issues();
	else
		logger_log_error("MSD_Handler - Error %d in state DEVICE_DETECTED. Ignoring the event.",result);
}

static void msd_enter_read_out_device_info(msd_uevent_info_cache_t *uevent_info_cache, device_t *device)
{
	int num_expected_part_cnt;
	logger_log_debug("MSD_Handler - Device %s: in state <read_out_device_info>",device_get_id(device));

	num_expected_part_cnt=msd_uevent_get_partition_cnt(uevent_info_cache);
	if (num_expected_part_cnt==0)
	{
		logger_log_debug("MSD_Handler - found no partition table but a file system directly on the device.");
		//default is 1 partition for device with no partition table
		device_fsm_enter_device_automounting(device,1);
	}
	else
	{
		logger_log_debug("MSD_Handler - found %d partitions on device.",num_expected_part_cnt);
		device_fsm_enter_device_automounting(device,num_expected_part_cnt);
	}
	//if we don't have any partitions, the device itself acts as partition.
	//It provides all info we are expecting from a partition as well (CDs are handled this way for instance)
	if (num_expected_part_cnt<1)
		msd_signal_partition_added(uevent_info_cache,device);
}

static void msd_enter_partition_detected(device_t *device,
		msd_uevent_info_cache_t *uevent_info_cache)
{
	//this attribute is copied into the model, so we don't have
	//to take care for its lifetime
	char mnt_point[PATH_MAX];

	partition_unsupported_reason_t unsupported_reason;
	partition_t *partition;

	unsupported_reason=msd_prepare_partition(device, &partition, uevent_info_cache);

	if (unsupported_reason==_SUPPORTED)
	{
		app_iface_signal_partition_detected(partition);
		logger_log_debug("MSD_Handler - Partition %s in state detected",partition_get_id(partition));
		if (msd_create_mount_point(mnt_point,PATH_MAX,uevent_info_cache)!=RESULT_OK)
		{
			// in case mnt_point == NULL, partition_fsm_enter_mounting will directly transition
			// to state PARTITION_MOUNT_ERR, which is appropriate for this case
			partition_fsm_enter_mounting(partition,NULL);
		}
		else
			partition_fsm_enter_mounting(partition,mnt_point);
	}
	else
		partition_fsm_enter_unsupported(partition, unsupported_reason);
}

static void msd_enter_partition_signal_dev_added(msd_uevent_info_cache_t *dev_uevent_info_cache,
		msd_uevent_info_cache_t *part_uevent_info_cache)
{
	const char *part_dev_node;
	device_t *device;
	msd_signal_device_added(dev_uevent_info_cache);

	part_dev_node=msd_uevent_get_dev_node(part_uevent_info_cache);

	device=device_list_find_device(msd_uevent_get_dev_node(dev_uevent_info_cache));
	if (device!=NULL)
	{
		if (device_get_state(device)==DEVICE_AUTOMOUNTING)
		{
			msd_enter_partition_detected(device,part_uevent_info_cache);
		}
		else
		{
			logger_log_error("MSD_Handler - The parent device of partition %s is not in state DEVICE_AUTOMOUNTING."
					" Ignoring the event.",part_dev_node);
		}
	}
	else
	{
		logger_log_error("MSD_Handler - Unable to add the parent device for partition %s. "
				"Ignoring the event.",part_dev_node);
	}
}

static void msd_enter_partition_signal_media_inserted(msd_uevent_info_cache_t *dev_uevent_info_cache,
		msd_uevent_info_cache_t *part_uevent_info_cache)
{
	const char *part_dev_node;
	device_t *device;
	msd_signal_device_media_inserted(dev_uevent_info_cache);

	part_dev_node=msd_uevent_get_dev_node(part_uevent_info_cache);

	device=device_list_find_device(msd_uevent_get_dev_node(dev_uevent_info_cache));
	if (device!=NULL)
	{
		if (device_get_state(device)==DEVICE_AUTOMOUNTING)
		{
			msd_enter_partition_detected(device,part_uevent_info_cache);
		}
		else
		{
			logger_log_error("MSD_Handler - Unable to bring the parent device into state DEVICE_AUTOMOUNTING. "
					"Ignoring the add event of partition: %s.",part_dev_node);
		}
	}
	else
	{
		logger_log_error("MSD_Handler - Signal_media_inserted called for a device that is not handled by "
				"the automounter yet. Implementation error. (part_id: %s)",part_dev_node);
	}
}
//--------------------------------------------------------------------------------------------------------

//--------------------------------------- others ---------------------------------------------------------
static void msd_prepare_device_changed_events(msd_uevent_info_cache_t *uevent_info_cache)
{
	if (msd_is_media_change_event(uevent_info_cache) ||
			msd_is_media_eject_complete_event(uevent_info_cache))
	{
		if (msd_device_contains_media(uevent_info_cache))
			msd_signal_device_media_inserted(uevent_info_cache);
		else
			msd_signal_device_media_removed(uevent_info_cache);
	}
}

static bool msd_is_media_change_event(msd_uevent_info_cache_t *uevent_info_cache)
{
	bool result;
	/* probe for DISK_MEDIA_CHANGE which should be 1 */
	const char *media_changed;

	media_changed = msd_uevent_get_value_from_udevice_property_or_default(uevent_info_cache,
									"DISK_MEDIA_CHANGE",NULL);
	if (media_changed == NULL)
		result = false;
	else
		result = strcmp(media_changed,"1") == 0;

	return result;
}

static bool msd_is_media_eject_complete_event(msd_uevent_info_cache_t *uevent_info_cache)
{
         bool result;
         /* probe for DISK_EJECT_COMPLETE which should be 1 */
         const char *eject_complete;

         eject_complete = msd_uevent_get_value_from_udevice_property_or_default(uevent_info_cache,
									"DISK_EJECT_COMPLETE",NULL);

	 if (eject_complete == NULL)
		result = false;
	 else
	        result = strcmp(eject_complete,"1") == 0;

         return result;
}
static bool msd_device_contains_media(msd_uevent_info_cache_t *uevent_info_cache)
{
	int part_cnt;
	long media_size;
	part_cnt=msd_uevent_get_partition_cnt(uevent_info_cache);
	media_size=msd_uevent_get_longvalue_sys_attr_or_default(uevent_info_cache,"size",-1);

	//To check whether device contains media,
	//udev attribute media_size should be more than zero and
	//one or more of the following conditions should be satisfied:
	//fs on board or partitions are present or its an audio cd
	return (msd_uevent_get_filesystem(uevent_info_cache) ||
		   part_cnt > 0								||
		   msd_uevent_contains_cdda(uevent_info_cache)) &&
		   (media_size>0);
}

static device_t *msd_find_device_in_model(msd_uevent_info_cache_t *uevent_info_cache)
{
	device_t *device;
	device=device_list_find_device(msd_uevent_get_dev_node(uevent_info_cache));
	return device;
}

static partition_t *msd_find_partition_in_model(msd_uevent_info_cache_t *uevent_info_cache)
{
	partition_t *partition;
	partition=device_list_find_partition_by_id(msd_uevent_get_dev_node(uevent_info_cache));
	return partition;
}

static partition_unsupported_reason_t msd_prepare_partition(device_t *device, partition_t **partition_ptr,
		msd_uevent_info_cache_t *uevent_info_cache)
{
	partition_unsupported_reason_t unsupported_reason=_SUPPORTED;
	error_code_t result;

	//this attribute is copied while putting it into the model within this function.
	//So we don't have to take care for its life time
	const char *part_dev_node;

	partition_metadata_t *metadata;

	part_dev_node=msd_uevent_get_dev_node(uevent_info_cache);

	if (blacklist_is_partition_blacklisted(part_dev_node))
	{
		logger_log_info("MSD_Handler - The partition %s is blacklisted. Marking it as unsupported.!",part_dev_node);
		result=msd_metadata_create_for_blacklisted_partition(&metadata,uevent_info_cache, &unsupported_reason);
	}
	else
		result=msd_metadata_create_for_partition(&metadata,uevent_info_cache, &unsupported_reason);

	if (result==RESULT_NORESOURCE)
		automounter_die_on_resource_issues();

	(*partition_ptr)=device_add_partition_by_device(device, part_dev_node, metadata);
	if ((*partition_ptr)==NULL)
		automounter_die_on_resource_issues();

	return unsupported_reason;
}

static  error_code_t msd_create_mount_point(char *mount_point, size_t mount_point_buffer_size,
		msd_uevent_info_cache_t *uevent_info_cache)
{
	const char *media_dir;
	char mp[PATH_MAX];
	bool leave_loop=false;

	error_code_t result=RESULT_OK;
	int mp_invalid_cntr=0;

	media_dir=configuration_get_media_dir();
	msd_metadata_create_mountpoint_from_partition(uevent_info_cache,mp,PATH_MAX);

	snprintf(mount_point,mount_point_buffer_size,"%s/AAM%s",media_dir,mp);

	while (!leave_loop)
	{
		if (mkdir(mount_point,configuration_get_mountpoint_mode())!=0)
		{
			//directory already there?
			if (errno!=EEXIST)
			{
				//seems we have a general problem with the  suggested mount point
				result=RESULT_INVALID_MOUNTPOINT;
				leave_loop=true;
			}
			else
				//-> create a new one with an underscore
				strcat(mount_point,"_");
		}
		else
			leave_loop=true;

		if (mp_invalid_cntr>configuration_mp_ext_give_up_cnt() && !leave_loop)
		{
			leave_loop=true;
			result=RESULT_INVALID_MOUNTPOINT;
		}
		else
			mp_invalid_cntr++;
	}

	return result;
}
//--------------------------------------------------------------------------------------------------------
